/*********************************************************************
 *
 *              dsPIC30F Auto Baud Source Code
 *
 *********************************************************************
 * FileName:        UART Auto Baud by Regression.c
 * Dependencies:    p30F6014.h
 *                  math.h
 * Date:            10/08/2004
 * Processor:       dsPIC30F6014
 * Complier:        MPLAB C30 1.20.02
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date       Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Mike Garbutt         10/8/2004  Original        (Rev 1.0)
 *
 ********************************************************************/

#include "p30F6014.h"            //Standard header file
#include "math.h"                //Math library
#define Fcy  29491200            //To allow calculation of baud rate for display

//---------------------------------------------------------------------------
//Prototypes

void SetupAutoBaud(void);        //Function to set up UART1, IC1 and TMR3
void CalculateBaud(void);        //Function to calculate the U1BRG value

//---------------------------------------------------------------------------
//Variables

unsigned int ICCount = 0;        //Count the number of Capture events
unsigned int T3Count = 0;        //Count the number of Timer 3 interrupts
unsigned int CurrentCapture;     //Record time of UART edge
unsigned int PreviousCapture;    //Store previous edge time measurement
unsigned int CaptureDifference;  //Difference between times of UART edges
long SumXY, SumY;                //Intermediate value for regression calculation
long RegressionData[10];         //Data to do linear regression
unsigned long BaudRate;          //Calculate the baud rate

//---------------------------------------------------------------------------
//Main routine
//Loops forever detecting the baud rate from incoming UART data of 0x55
//and outputing a message each time the baud rate is calculated.

int main(void)
{
   while(1)                                  //Loop forever
   {
      SetupAutoBaud();                       //Set up UART1, IC1 and TMR3 for autobaud
      while(U1BRG == 0) {}                   //Wait for autobaud to complete
      BaudRate = (Fcy / 16) / (U1BRG + 1);   //See what baud rate is being used
      printf("Baud rate: %ld\r", BaudRate);  //Output text with the baud rate
      while(U1STAbits.TRMT == 0) {}          //Wait for transmission to complete
   }
}	                                          //End of main()

//---------------------------------------------------------------------------
//Set up the peripherals and interrupts to do baud rate detection

void SetupAutoBaud(void)
{
   U1BRG = 0;                 //U1BRG initially unknown
   U1MODE = 0x8020;           //Enable auto baud detection in UART
   U1STA = 0x0000;            //Set up rest of UART to default state
 
   ICCount = 0;               //Initialize the number of Capture events
   IC1CON = 0x0000;           //Reset Input Capture 1 module
   IC1CON = 0x0001;           //Enable Input Capture 1 module
   IFS0bits.IC1IF = 0;        //Clear Capture 1 interrupt flag
   IEC0bits.IC1IE = 1;        //Enable Capture 1 interrupt

   T3CON = 0x0000;            //Timer 3 off
   IEC0bits.T3IE = 0;         //Clear Timer 3 interrupt enable
   T3Count = 0;               //Initialize the number of Timer 3 interrupts
   PR3 = 0xffff;              //Timer 3 period is maximum 
   T3CON = 0x8000;            //Timer 3 on with 1:1 prescaler and internal clock                
}

//---------------------------------------------------------------------------
//Calculate value for U1BRG baud rate generator

void CalculateBaud(void)
{
   int i;                                 //Index to sum the errors 
   long Slope;                            //Slope (bit time) of regression
   long Yintercept;                       //Expected (calculated) time of first edge
   long PlotLine;                         //Expected (calculated) time of each edge
   long SumOfErrors = 0;                  //Sum of all the errors |Measured-Expected|

   Slope = (2 * SumXY - 9 * SumY) / 165;  //Calculate slope = one bit time
   Yintercept = (SumY - Slope * 45) / 10; //Calculate expected time of first edge

   PlotLine = Yintercept;                 //Initialize expected time of each edge
   for(i=0; i<10; i++)                    //Loop to add all the absolute errors
   {
      SumOfErrors += abs(RegressionData[i] - PlotLine);  //Calculate and add next error
      PlotLine += Slope;                  //Calculate expected time of next edge
   }

   if((SumOfErrors * 2) < Slope)          //Check if mean absolute error < 5%
   {                                      //(is twice sum of 10 errors < one bit time?)
      U1BRG = ((Slope + 8) >> 4) - 1;       //Calculate UxBRG (rounding by adding one half)
      U1MODE = 0x8000;                    //Enable UART and disable auto baud detection
      U1STA = 0x0400;                     //Enable transmission
   }
   else
   {
      SetupAutoBaud();                    //Error too large so start over
   }
}

//---------------------------------------------------------------------------
//Input Capture 1 ISR
//Gets time measurements and adds to the sums needed for regression

void _ISR _IC1Interrupt(void)
{
   IFS0bits.IC1IF = 0;                 //Clear Capture 1 interrupt flag
   PreviousCapture = CurrentCapture;   //Store previous time measurement 
   CurrentCapture = IC1BUF;            //Get new time measurement
   T3Count = 0;                        //Reset the timeout counter
   if(ICCount == 0)                    //Check if first edge
   {
      IFS0bits.T3IF = 0;               //Clear Timer 3 interrupt flag
      IEC0bits.T3IE = 1;               //Enable Timer 3 interrupt for timeout check
      RegressionData[0] = 0;           //Initial value for time of first edge
      SumY = 0;                        //Initial value for sum of time measurements
      SumXY = 0;                       //Initial value for sum of bit number x time
   }  
   else                                //Check if not first edge
   {
      CaptureDifference = CurrentCapture - PreviousCapture; //Get time difference
      RegressionData[ICCount] = RegressionData[ICCount-1] + CaptureDifference;
                                       //Add time difference to prevous time measurement
      SumY += RegressionData[ICCount]; //Sum the time measurements
      SumXY += RegressionData[ICCount] * ICCount; //Sum the bit number x time measurement
   }
   ICCount++;                          //Increment count of edges
   if(ICCount == 10)                   //Check if last edge
   {
      IEC0bits.IC1IE = 0;              //Clear Capture 1 interrupt enable
      IEC0bits.T3IE = 0;               //Clear Timer 3 interrupt enable
      CalculateBaud();                 //Calculate the U1BRG value and enable the UART
   }  
}

//---------------------------------------------------------------------------
//Timer 3 ISR
//Check for timeout indicated by two rollovers since previous input capture

void _ISR _T3Interrupt(void)
{
   IFS0bits.T3IF = 0;            //Clear Timer 3 interrupt flag
   T3Count++;                    //Increment count of interrupts since last capture
   if(T3Count == 2)              //Check for too many timer rollovers since last capture 
   {
      SetupAutoBaud();           //Timeout so start over
   }
}
